// Slider functionality
let currentSlide = 0
const slides = document.querySelectorAll('.slide-lux')
const dots = document.querySelectorAll('.dot-lux')
const totalSlides = slides.length

function goToSlide(index) {
  if (slides.length === 0) return
  
  slides[currentSlide]?.classList.remove('active')
  dots[currentSlide]?.classList.remove('active')
  
  currentSlide = index
  if (currentSlide >= totalSlides) currentSlide = 0
  if (currentSlide < 0) currentSlide = totalSlides - 1
  
  slides[currentSlide]?.classList.add('active')
  dots[currentSlide]?.classList.add('active')
}

function goToNext() {
  goToSlide(currentSlide + 1)
}

function goToPrevious() {
  goToSlide(currentSlide - 1)
}

// Auto-slide
if (slides.length > 0) {
  setInterval(() => {
    goToNext()
  }, 3000)
  
  goToSlide(0)
}

// Initialize page
function initPage() {
  const currentPage = window.location.pathname.includes('rent.html') ? 'rent' : 'index'
  
  if (currentPage === 'rent') {
    initRentPage()
  } else {
    initIndexPage()
  }
}

// Index page functions
function initIndexPage() {
  loadRentals()
  updateStats()
}

function loadRentals() {
  const saved = localStorage.getItem('campingRentals')
  let rentals = []
  
  if (saved) {
    try {
      rentals = JSON.parse(saved)
    } catch (e) {
      rentals = []
    }
  } else {
    rentals = [
      {
        id: 1,
        item: 'Палатка 4-местная',
        category: 'tent',
        date: '2025-06-15',
        review: null
      },
      {
        id: 2,
        item: 'Спальник зимний',
        category: 'sleeping',
        date: '2025-05-20',
        review: {
          rating: 5,
          text: 'Превосходное качество. Очень доволен покупкой.'
        }
      },
      {
        id: 3,
        item: 'Рюкзак 70л',
        category: 'backpack',
        date: '2025-04-10',
        review: null
      }
    ]
    localStorage.setItem('campingRentals', JSON.stringify(rentals))
  }
  
  renderRentals(rentals)
}

function renderRentals(rentals) {
  const container = document.getElementById('rentals-container')
  if (!container) return
  
  if (rentals.length === 0) {
    container.innerHTML = `
      <div class="empty-lux">
        <div class="empty-icon-lux">
          <img src="icons/tent.svg" alt="Tent" width="48" height="48">
        </div>
        <h2 class="empty-title-lux">Нет аренд</h2>
        <p class="empty-text-lux">Начните с аренды премиум оборудования</p>
        <button class="empty-button-lux" onclick="window.location.href='rent.html'">
          Арендовать
        </button>
      </div>
    `
    return
  }
  
  const categoryIcons = {
    tent: 'tent.svg',
    sleeping: 'sleeping-bag.svg',
    backpack: 'backpack.svg'
  }
  
  container.innerHTML = rentals.map((rental, index) => {
    const icon = categoryIcons[rental.category] || 'tent.svg'
    const date = new Date(rental.date).toLocaleDateString('ru-RU', { 
      day: 'numeric', 
      month: 'long', 
      year: 'numeric' 
    })
    
    const reviewSection = rental.review 
      ? `
        <div class="review-display-lux">
          <div class="review-stars-lux">
            ${Array.from({ length: 5 }).map((_, i) => 
              `<img src="icons/star.svg" alt="Star" width="16" height="16" style="${i < rental.review.rating ? 'filter: brightness(0) saturate(100%) invert(70%) sepia(95%) saturate(1352%) hue-rotate(358deg) brightness(101%) contrast(101%);' : 'opacity: 0.3;'}">`
            ).join('')}
          </div>
          <p class="review-text-lux">${rental.review.text}</p>
        </div>
      `
      : `
        <div class="review-form-lux">
          <div class="review-header-lux">
            <img src="icons/message-square.svg" alt="Message" width="18" height="18">
            <span>Оставить отзыв</span>
          </div>
          <div class="rating-lux" id="rating-${rental.id}">
            ${Array.from({ length: 5 }).map((_, i) => 
              `<button type="button" class="star-btn-lux" onclick="setRating(${rental.id}, ${i + 1})">
                <img src="icons/star.svg" alt="Star" width="20" height="20" id="star-${rental.id}-${i + 1}">
              </button>`
            ).join('')}
          </div>
          <textarea
            class="review-textarea-lux"
            id="review-text-${rental.id}"
            placeholder="Поделитесь впечатлениями..."
            rows="3"
          ></textarea>
          <button
            class="submit-review-lux"
            onclick="submitReview(${rental.id})"
            id="submit-review-${rental.id}"
          >
            Отправить
          </button>
        </div>
      `
    
    return `
      <div class="rental-item-lux">
        <div class="rental-header-lux">
          <div class="rental-icon-lux">
            <img src="icons/${icon}" alt="${rental.category}" width="24" height="24">
          </div>
          <div class="rental-info-lux">
            <h3 class="rental-name-lux">${rental.item}</h3>
            <div class="rental-date-lux">
              <img src="icons/calendar.svg" alt="Calendar" width="16" height="16">
              <span>${date}</span>
            </div>
          </div>
          <div class="rental-number-lux">#${rental.id}</div>
        </div>
        ${reviewSection}
      </div>
    `
  }).join('')
}

function updateStats() {
  const saved = localStorage.getItem('campingRentals')
  let rentals = []
  
  if (saved) {
    try {
      rentals = JSON.parse(saved)
    } catch (e) {
      rentals = []
    }
  }
  
  const tentCount = rentals.filter(r => r.category === 'tent').length
  const sleepingCount = rentals.filter(r => r.category === 'sleeping').length
  const backpackCount = rentals.filter(r => r.category === 'backpack').length
  const totalCount = rentals.length
  
  document.getElementById('tent-count').textContent = tentCount
  document.getElementById('sleeping-count').textContent = sleepingCount
  document.getElementById('backpack-count').textContent = backpackCount
  document.getElementById('total-count').textContent = totalCount
}

let currentRatings = {}

function setRating(rentalId, rating) {
  currentRatings[rentalId] = rating
  
  for (let i = 1; i <= 5; i++) {
    const star = document.getElementById(`star-${rentalId}-${i}`)
    if (star) {
      if (i <= rating) {
        star.style.filter = 'brightness(0) saturate(100%) invert(70%) sepia(95%) saturate(1352%) hue-rotate(358deg) brightness(101%) contrast(101%);'
      } else {
        star.style.filter = 'opacity(0.3)'
      }
    }
  }
  
  const submitBtn = document.getElementById(`submit-review-${rentalId}`)
  const reviewText = document.getElementById(`review-text-${rentalId}`)
  
  if (submitBtn && reviewText) {
    submitBtn.disabled = !currentRatings[rentalId] || !reviewText.value.trim()
  }
}

function submitReview(rentalId) {
  const saved = localStorage.getItem('campingRentals')
  let rentals = []
  
  if (saved) {
    try {
      rentals = JSON.parse(saved)
    } catch (e) {
      rentals = []
    }
  }
  
  const reviewText = document.getElementById(`review-text-${rentalId}`)?.value || ''
  const rating = currentRatings[rentalId] || 0
  
  if (!rating || !reviewText.trim()) return
  
  const updated = rentals.map(rental => {
    if (rental.id === rentalId) {
      return {
        ...rental,
        review: {
          rating: rating,
          text: reviewText
        }
      }
    }
    return rental
  })
  
  localStorage.setItem('campingRentals', JSON.stringify(updated))
  delete currentRatings[rentalId]
  loadRentals()
  updateStats()
}

// Rent page functions
function initRentPage() {
  renderCategories()
  
  const dateInput = document.getElementById('date-input')
  if (dateInput) {
    const today = new Date()
    const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
    dateInput.min = today.toISOString().split('T')[0]
    dateInput.max = maxDate.toISOString().split('T')[0]
  }
  
  const form = document.getElementById('rent-form')
  if (form) {
    form.addEventListener('submit', handleSubmit)
  }
}

const categories = [
  { 
    id: 'tent', 
    name: 'Палатки', 
    icon: 'tent.svg',
    description: 'Премиум палатки для комфортного отдыха на природе',
    items: ['2-местные', '4-местные', '6-местные', 'Семейные']
  },
  { 
    id: 'sleeping', 
    name: 'Спальные системы', 
    icon: 'sleeping-bag.svg',
    description: 'Элитные спальники и аксессуары для идеального сна',
    items: ['Летние', 'Трехсезонные', 'Зимние', 'Экстремальные']
  },
  { 
    id: 'kitchen', 
    name: 'Кухня', 
    icon: 'utensils-crossed.svg',
    description: 'Профессиональное оборудование для приготовления',
    items: ['Горелки', 'Посуда', 'Холодильники', 'Аксессуары']
  },
  { 
    id: 'navigation', 
    name: 'Навигация', 
    icon: 'compass.svg',
    description: 'Точные навигационные приборы и устройства',
    items: ['GPS-навигаторы', 'Компасы', 'Карты', 'Рации']
  }
]

let selectedCategory = ''

function renderCategories() {
  const container = document.getElementById('categories-container')
  if (!container) return
  
  container.innerHTML = categories.map(category => {
    return `
      <button 
        type="button" 
        class="category-item-lux" 
        data-category="${category.id}"
        onclick="selectCategory('${category.id}')"
      >
        <div class="category-icon-wrapper-lux">
          <img src="icons/${category.icon}" alt="${category.name}" width="24" height="24">
        </div>
        <div class="category-content-lux">
          <h3 class="category-name-lux">${category.name}</h3>
          <p class="category-desc-lux">${category.description}</p>
          <div class="category-tags-lux">
            ${category.items.map(item => 
              `<span class="category-tag-lux">${item}</span>`
            ).join('')}
          </div>
        </div>
        <div class="check-mark-lux" id="check-${category.id}" style="display: none;">
          <img src="icons/check-circle.svg" alt="Check" width="24" height="24">
        </div>
      </button>
    `
  }).join('')
}

function selectCategory(categoryId) {
  selectedCategory = categoryId
  
  // Update UI
  document.querySelectorAll('.category-item-lux').forEach(card => {
    card.classList.remove('selected')
  })
  
  document.querySelectorAll('.check-mark-lux').forEach(badge => {
    badge.style.display = 'none'
  })
  
  const selectedCard = document.querySelector(`[data-category="${categoryId}"]`)
  if (selectedCard) {
    selectedCard.classList.add('selected')
    const checkMark = document.getElementById(`check-${categoryId}`)
    if (checkMark) {
      checkMark.style.display = 'block'
    }
  }
  
  // Show date section
  const dateSection = document.getElementById('date-section')
  if (dateSection) {
    dateSection.style.display = 'block'
  }
  
  updateSubmitButton()
}

function updateSubmitButton() {
  const submitBtn = document.getElementById('submit-btn')
  const dateInput = document.getElementById('date-input')
  
  if (submitBtn && dateInput) {
    submitBtn.disabled = !selectedCategory || !dateInput.value
  }
}

function handleSubmit(e) {
  e.preventDefault()
  
  if (!selectedCategory) return
  
  const dateInput = document.getElementById('date-input')
  if (!dateInput || !dateInput.value) return
  
  const saved = localStorage.getItem('campingRentals')
  const rentals = saved ? JSON.parse(saved) : []
  
  const categoryData = categories.find(c => c.id === selectedCategory)
  
  const newRental = {
    id: Date.now(),
    item: `${categoryData.name} - ${categoryData.items[0]}`,
    category: selectedCategory,
    date: dateInput.value,
    review: null
  }
  
  rentals.push(newRental)
  localStorage.setItem('campingRentals', JSON.stringify(rentals))
  
  const successMessage = document.getElementById('success-message')
  if (successMessage) {
    successMessage.style.display = 'flex'
  }
  
  setTimeout(() => {
    window.location.href = 'index.html'
  }, 2000)
}

// Initialize on page load
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initPage)
} else {
  initPage()
}

// Update submit button when date changes
const dateInput = document.getElementById('date-input')
if (dateInput) {
  dateInput.addEventListener('change', updateSubmitButton)
}

